<?php

namespace App\View\Components;

use App\Repositories\StatsRepository;
use Illuminate\Contracts\View\View;
use Illuminate\View\Component;

class Widget extends Component
{
    private const WIDGET_TYPES = [
        'user' => 'Total Users',
        'destination' => 'Total Destinations',
        'package' => 'Total Packages',
        'booking' => 'Total Bookings',
        'role' => 'Total Roles',
        'pending-booking' => 'Pending Bookings',
        'approved-booking' => 'Approved Bookings',
        'completed-booking' => 'Completed Bookings',
        'rejected-booking' => 'Rejected Bookings',
    ];

    private const TYPES_WITH_CONDITIONS = [
        'package',
        'booking',
        'pending-booking',
        'approved-booking',
        'completed-booking',
        'rejected-booking'
    ];

    public string $title;
    public int $currentValue;
    public float $percentageChange;
    public array $chartData;
    private array $conditions = [];

    public function __construct(
        private StatsRepository $stats,
        public string $type,
        public string $color
    ) {
        $this->setUserConditions();
        $this->title = self::WIDGET_TYPES[$type] ?? 'Unknown';
        $this->loadData();
    }

    private function setUserConditions(): void
    {
        if (!in_array($this->type, self::TYPES_WITH_CONDITIONS)) {
            return;
        }

        if (user()->hasRole('agent')) {
            $this->conditions['agent_id'] = auth()->id();
        }
        if (user()->hasRole('user')) {
            $this->conditions['user_id'] = auth()->id();
        }
    }

    private function loadData(): void
    {
        $this->currentValue = $this->getCurrentCount();
        $this->percentageChange = $this->calculatePercentageChange();
        $this->chartData = $this->getLastSevenDaysData();
    }

    private function getCurrentCount(): int
    {
        $table = $this->getTableName();
        $conditions = $this->getConditions();
        return $this->stats->getCount($table, $conditions);
    }

    private function calculatePercentageChange(): float
    {
        $currentCount = $this->currentValue;
        $lastWeekCount = $this->stats->getCount(
            $this->getTableName(),
            array_merge(
                $this->conditions,
                ['created_at' => [now()->subWeeks(2), now()->subWeek()]]
            )
        );

        if ($lastWeekCount === 0) {
            return 0.0;
        }

        return (($currentCount - $lastWeekCount) / $lastWeekCount) * 100;
    }

    private function getLastSevenDaysData(): array
    {
        $data = [];
        $table = $this->getTableName();

        for ($i = 6; $i >= 0; $i--) {
            $date = now()->subDays($i);
            $data[] = $this->stats->getDailyStats(
                $table,
                $date->toDateString(),
                $this->getConditions()
            );
        }

        return $data;
    }

    private function getTableName(): string
    {
        return match ($this->type) {
            'user' => 'users',
            'package' => 'packages',
            'role' => 'roles',
            'booking', 'pending-booking', 'approved-booking',
            'completed-booking', 'rejected-booking' => 'bookings',
            'destination' => 'destinations',
            default => throw new \InvalidArgumentException("Invalid widget type: {$this->type}")
        };
    }

    private function getConditions(): array
    {
        if ($this->type === 'destination') {
            return [];
        }

        if (str_contains($this->type, 'booking') && $this->type !== 'booking') {
            return array_merge(
                $this->conditions,
                ['status' => str_replace('-booking', '', $this->type)]
            );
        }

        return $this->conditions;
    }

    public function render(): View
    {
        return view('components.widget');
    }
}
